using ConnectorLib.JSON;
using Inventory;

namespace CrowdControl.Delegates.Effects.Implementations
{
    [Effect("transmute_all_copper")]
    public class Transmute : Effect
    {
        public Transmute(CrowdControl mod, NetworkClient client) : base(mod, client) { }
        
        public override EffectResponse Start(EffectRequest request)
        {
            PlayerController player = Manager.main.player;
            if (player == null) 
                return EffectResponse.Failure(request.ID, StandardErrors.PlayerNotFound);

            InventoryHandler inventory = player.playerInventoryHandler;
            if (inventory == null)
                return EffectResponse.Failure(request.ID, StandardErrors.PlayerNotFound, "Player inventory not found.");

            bool transmutedAny = false;
            int invMax = inventory.maxSize;
            for (int i = 0; i < invMax; i++)
            {
                ContainedObjectsBuffer items = inventory.GetContainedObjectData(i);
                ObjectID itemID = items.objectID;

                if (itemID is
                    ObjectID.TinOre or ObjectID.IronOre or
                    ObjectID.GoldOre or ObjectID.ScarletOre or ObjectID.OctarineOre or
                    ObjectID.GalaxiteOre or ObjectID.SolariteOre or ObjectID.PandoriumOre)
                {
                    TransmuteItem(inventory, i, items, ObjectID.CopperOre);
                    transmutedAny = true;
                }

                else if (itemID is
                         ObjectID.TinBar or ObjectID.IronBar or
                         ObjectID.GoldBar or ObjectID.ScarletBar or ObjectID.OctarineBar or
                         ObjectID.GalaxiteBar or ObjectID.SolariteBar or ObjectID.PandoriumBar)
                {
                    TransmuteItem(inventory, i, items, ObjectID.CopperBar);
                    transmutedAny = true;
                }
            }

            return transmutedAny
                ? EffectResponse.Success(request.ID)
                : EffectResponse.Failure(request.ID, StandardErrors.NoValidTargets, "No transmutable metals found in inventory.");
        }
        
        private static void TransmuteItem(InventoryHandler inventory, int slotIndex, ContainedObjectsBuffer items, ObjectID newItemID)
        {
            ObjectID itemID = items.objectID;
            int amount = items.amount;
            inventory.DestroyObject(Manager.main.player, slotIndex, itemID);
            Manager.main.player.QueueInputAction(new()
            {
                action = UIInputAction.InventoryChange,
                inventoryChangeData = Create.CreateItem(
                    inventory.inventoryEntity,
                    slotIndex,
                    newItemID,
                    amount,
                    EntityMonoBehaviour.ToWorldFromRender(inventory.entityMonoBehaviour.RenderPosition),
                    0)
            });
        }
    }
}